/* $Id: i2c.c,v 1.10 1999/01/26 19:45:10 ericb Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* I2C test access */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "e1432.h"

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: i2c.c,v 1.10 1999/01/26 19:45:10 ericb Exp $";
static const char *progname;

static int
init(SHORTSIZ16 laddr, E1432ID *hw)
{
    struct e1432_hwconfig hwconfig;
    int     status;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    status = e1432_get_hwconfig(1, &laddr, &hwconfig);
    CHECK(e1432_print_errors(1));
    if (status < 0)
	CHECK(e1432_install(1, &laddr, 0, "/opt/e1432/lib/sema.bin"));
    CHECK(e1432_assign_channel_numbers(1, &laddr, hw));

    return 0;
}

static int
run(E1432ID hw, int chan, int addr, int doread, int value)
{
    SHORTSIZ16 resp;

    if (doread)
    {
	CHECK(e1432_read_i2c(hw, chan, addr, &resp));
	(void) printf("got response = 0x%2.2x\n", resp);
    }
    else
	CHECK(e1432_write_i2c(hw, chan, addr, value));

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-ruV] [-a i2c_addr] [-c chan]\n"
		   "\t[-l laddr] [-v value]\n"
		   "\t-a: Use I2C address <i2c_addr> (default 0)\n"
		   "\t-c: Use I2C on channel <chan> (default 1)\n"
		   "\t-L: Use E1432 at logical address <laddr> (default 8)\n"
		   "\t-r: Do read instead of write (default write)\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Write <value> (default 0)\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    SHORTSIZ16 laddr = 8;
    E1432ID hw;
    char   *p;
    int     c, addr, chan, doread, value;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    addr = 0;
    chan = 1;
    doread = 0;
    laddr = 8;
    value = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "a:c:L:ruv:V")) != -1)
	switch (c)
	{
	case 'a':
	    addr = strtol(optarg, &p, 0);
	    if (optarg == p)
	    {
		(void) fprintf(stderr,
			       "%s: invalid I2C address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'c':
	    chan = strtol(optarg, &p, 0);
	    if (optarg == p)
	    {
		(void) fprintf(stderr,
			       "%s: invalid channel: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'L':
	    laddr = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'r':
	    doread = 1;
	    break;
	case 'v':
	    value = strtol(optarg, &p, 0);
	    if (optarg == p)
	    {
		(void) fprintf(stderr,
			       "%s: invalid value: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    if (init(laddr, &hw) < 0)
	return EXIT_FAILURE;
    if (run(hw, chan, addr, doread, value) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
